//##########################################
//  UV Integrator
//  Author: Zac Levinson
//  Version: 1.0
//
//  Integrate UV radiation via a VEML6070
//  I2C UV sensor. The following external
//  libraries are required:
//    - Adafruit VEML6070
//
//  The LCD should be connected to digital
//  pins 5-10 as follows:
//    -Pin 5: RS
//    -Pin 6: EN
//    -Pin 7-10: DB4-DB7
//
//  The start button should be connected
//  between digital pin 2 and ground.
//
//  The VEML6070 is a very linear sensor
//  and returns a 16-bit number corresponding
//  to UV irradiance. We have five characters,
//  including a decimal point, to display the
//  value. To accomodate this, the sensor output
//  has been normalized to a percentage of 65535.
//  
//  If the start button is pressed and held for
//  five seconds the time and dose will reset
//  to zero.
//############################################

#include <Wire.h>
#include "Adafruit_VEML6070.h"
#include <LiquidCrystal.h>
#include <avr/power.h>

// Initialize LCD display and UV sensor
LiquidCrystal lcd(5, 6, 7, 8, 9, 10);
Adafruit_VEML6070 uv = Adafruit_VEML6070();

//Define variables for program state and start/stop button pin
volatile int state = 0;
const int button_pin = 2;
int button_state = 1;

//Define LCD dimensions
const int LCD_CHAR = 16;
const int LCD_ROW = 2;

// Define variables for start time and operating duration
long start = 0;
long t = 0;
long dt = 0;
int minutes = 0;
int seconds = 0;

// Define variables for timing of LCD update and sensor reading
const int check = 500;
const int update_t = 1000;
long last_check = 0;
long last_update = 0;
long last_press = 0;

// Define variables for current irradiance and dose
// The irradiance is normalized to be a percentage of
// the maximum sensor output, or 2^16-1
double I = 0;
double E = 0;
const double I_norm = (double) 100/65535;
static char Istr[5];
static char Estr[6];

void setup(){
  // Disable some modules to try to save power
  power_adc_disable();
  power_spi_disable();
  power_usart0_disable();
  
  // Setup interrupt for start/stop button
  digitalWrite(button_pin, INPUT_PULLUP);

  // Initialize LCD
  lcd.begin(LCD_CHAR, LCD_ROW);
  lcd.print("T 00m00s I 0.000");
  lcd.setCursor(0, 1);
  lcd.print("Dose 0.0000");

  // Initiliaze the UV sensor and set integration time to 500ms
  uv.begin(VEML6070_4_T);
}

void loop() {
  check_button();
  
  if (state == 1){ // if we're measuring dose:
    //Compute the run time
    t = (millis() - start) + dt;
    minutes = t/60000;
    seconds = t/1000 - minutes*60;

    // Check UV sensor if it's been longer than
    // the integration time since the last reading
    if (millis() - last_check > check){
      I = (double) uv.readUV() * I_norm;
      last_check = millis();
      E += I/2;
    }
  }
  else{ //If we're not running, still measure UV
    I = (double) uv.readUV() * I_norm;
  }

  update_LCD();
}

void check_button(){
  long current_press = millis();
  int current_state = digitalRead(button_pin);
  long delta = current_press - last_press;

  if (button_state == LOW && delta > 5000){
    state = -1;
    minutes = 0;
    seconds = 0;
    E = 0;
    t = 0;
    dt = 0;
  }

  if (delta > 200){
    if (button_state == LOW && current_state == HIGH){
      state = (state + 1) % 2;

      if (state == 0){
        dt = t;
      }
      else{
        start = current_press;
      }
      button_state = current_state;
      last_press = current_press;
    }
    else if(button_state == HIGH && current_state == LOW){
      button_state = current_state;
      last_press = current_press;
    }
  }
}

void update_LCD(){
  // Update the LCD display
  if (millis() - last_update > update_t){
    last_update = millis();

    // Update run-time
    lcd.setCursor(2, 0);
    if (minutes < 10){
      lcd.print("0");
    }
    lcd.print(minutes);

    lcd.setCursor(5, 0);
    if (seconds < 10){
      lcd.print("0");
    }
    lcd.print(seconds);

    // Update dose
    lcd.setCursor(5,1);
    dtostrf(E, 6, 4, Estr);
    lcd.print(Estr);
  }

  // Update LCD with irradiance
  lcd.setCursor(11,0);
  dtostrf(I, 5, 3, Istr);
  lcd.print(Istr);
  
  // Blink a cursor if we're running
  lcd.setCursor(15,1);
  if (state == 1 && seconds % 2 == 0){
    lcd.print((char)255);
  }
  else{
    lcd.print((char)32);
  }
}

